/*
 * Decompiled with CFR 0.152.
 */
package org.apache.couchdb.nouveau.core;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.dropwizard.lifecycle.Managed;
import jakarta.validation.constraints.PositiveOrZero;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.Response;
import java.io.File;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.couchdb.nouveau.api.IndexDefinition;
import org.apache.couchdb.nouveau.core.IOUtils;
import org.apache.couchdb.nouveau.core.Index;
import org.apache.couchdb.nouveau.core.StripedLock;
import org.apache.couchdb.nouveau.lucene9.Lucene9AnalyzerFactory;
import org.apache.couchdb.nouveau.lucene9.Lucene9Index;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.misc.store.DirectIODirectory;
import org.apache.lucene.search.SearcherFactory;
import org.apache.lucene.search.SearcherManager;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class IndexManager
implements Managed {
    private static final Logger LOGGER = LoggerFactory.getLogger(IndexManager.class);
    @PositiveOrZero
    private int maxIndexesOpen;
    @PositiveOrZero
    private int commitIntervalSeconds;
    @PositiveOrZero
    private int idleSeconds;
    private Path rootDir;
    private ObjectMapper objectMapper;
    private ScheduledExecutorService schedulerExecutorService;
    private SearcherFactory searcherFactory;
    private LRUMap cache;
    private StripedLock<String> createLock;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public <R> R with(String name, IndexFunction<Index, R> indexFun) throws IOException, InterruptedException {
        this.evictIfOverCapacity();
        block17: while (true) {
            IndexHolder holder;
            if (!this.exists(name)) {
                throw new WebApplicationException("Index does not exist", Response.Status.NOT_FOUND);
            }
            LRUMap lRUMap = this.cache;
            synchronized (lRUMap) {
                holder = this.cache.computeIfAbsent(name, k -> new IndexHolder());
            }
            holder.lock.readLock().lock();
            if (holder.state == HolderState.NOT_LOADED || holder.state == HolderState.LOADED && !holder.index.isOpen()) {
                holder.lock.readLock().unlock();
                holder.lock.writeLock().lock();
                try {
                    if (holder.state == HolderState.LOADED && !holder.index.isOpen()) {
                        LOGGER.info("removing closed index {}", (Object)name);
                        holder.state = HolderState.UNLOADED;
                        holder.index = null;
                        LRUMap lRUMap2 = this.cache;
                        synchronized (lRUMap2) {
                            this.cache.remove(name, holder);
                            continue;
                        }
                    }
                    if (holder.state == HolderState.NOT_LOADED) {
                        holder.index = this.load(name);
                        holder.commitFuture = this.schedulerExecutorService.scheduleWithFixedDelay(this.commitFun(name, holder), this.commitIntervalSeconds, this.commitIntervalSeconds, TimeUnit.SECONDS);
                        holder.state = HolderState.LOADED;
                    }
                    holder.lock.readLock().lock();
                }
                finally {
                    holder.lock.writeLock().unlock();
                    continue;
                }
            }
            try {
                switch (holder.state.ordinal()) {
                    case 0: {
                        throw new IllegalStateException();
                    }
                    case 2: {
                        Thread.sleep(1000L);
                        continue block17;
                    }
                    case 1: {
                        R r = indexFun.apply(holder.index);
                        return r;
                    }
                }
                continue;
            }
            finally {
                holder.lock.readLock().unlock();
                continue;
            }
            break;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void evictIfOverCapacity() throws IOException, InterruptedException {
        while (true) {
            String candidate;
            LRUMap lRUMap = this.cache;
            synchronized (lRUMap) {
                if (this.cache.size() <= this.maxIndexesOpen) {
                    return;
                }
                candidate = (String)this.cache.entrySet().iterator().next().getKey();
            }
            this.unload(candidate, false);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void unload(String name, boolean forceDelete) throws IOException, InterruptedException {
        IndexHolder holder;
        LRUMap lRUMap = this.cache;
        synchronized (lRUMap) {
            holder = this.cache.computeIfAbsent(name, k -> new IndexHolder());
        }
        holder.lock.writeLock().lock();
        try {
            switch (holder.state.ordinal()) {
                case 1: {
                    if (forceDelete) {
                        holder.index.setDeleteOnClose(true);
                    }
                    LOGGER.info("closing {}", (Object)name);
                    try {
                        this.close(name, holder);
                    }
                    catch (IOException e) {
                        LOGGER.error("I/O exception when evicting {}", (Object)name, (Object)e);
                    }
                    holder.state = HolderState.UNLOADED;
                    holder.index = null;
                    return;
                }
                case 0: 
                case 2: {
                    if (!forceDelete) return;
                    IOUtils.rm(this.indexRootPath(name));
                    return;
                }
            }
            return;
        }
        finally {
            lRUMap = this.cache;
            synchronized (lRUMap) {
                this.cache.remove(name, holder);
            }
            holder.lock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void create(String name, IndexDefinition indexDefinition) throws IOException {
        if (this.exists(name)) {
            this.assertSame(indexDefinition, this.loadIndexDefinition(name));
            return;
        }
        Lock lock = this.createLock.writeLock(name);
        lock.lock();
        try {
            if (this.exists(name)) {
                this.assertSame(indexDefinition, this.loadIndexDefinition(name));
                return;
            }
            Path dstFile = this.indexDefinitionPath(name);
            Files.createDirectories(dstFile.getParent(), new FileAttribute[0]);
            Path tmpFile = Files.createTempFile(dstFile.getParent(), null, null, new FileAttribute[0]);
            boolean success = false;
            try {
                this.objectMapper.writeValue(tmpFile.toFile(), (Object)indexDefinition);
                Files.move(tmpFile, dstFile, StandardCopyOption.ATOMIC_MOVE);
                success = true;
            }
            finally {
                if (!success) {
                    Files.delete(tmpFile);
                }
            }
        }
        finally {
            lock.unlock();
        }
    }

    public boolean exists(String name) {
        return Files.exists(this.indexDefinitionPath(name), new LinkOption[0]);
    }

    public void deleteAll(String path, List<String> exclusions) throws IOException {
        LOGGER.info("deleting indexes matching {} (excluding {})", (Object)path, exclusions == null || exclusions.isEmpty() ? "nothing" : exclusions);
        String[] parts = path.split("/");
        this.deleteAll(this.rootDir, parts, 0, exclusions);
    }

    private void deleteAll(Path path, String[] parts, int index, List<String> exclusions) throws IOException {
        if (index == parts.length - 1) {
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(path, parts[index]);){
                stream.forEach(p -> {
                    if (exclusions != null && exclusions.indexOf(p.getFileName().toString()) != -1) {
                        return;
                    }
                    String relativeName = this.rootDir.relativize((Path)p).toString();
                    try {
                        this.deleteIndex(relativeName);
                    }
                    catch (IOException | InterruptedException e) {
                        LOGGER.error("Exception deleting {}", p, (Object)e);
                    }
                    do {
                        File f;
                        if (!(f = p.toFile()).isDirectory() || f.list().length != 0) continue;
                        f.delete();
                    } while ((p = p.getParent()) != null && !this.rootDir.equals(p));
                });
            }
            return;
        }
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(path, parts[index]);){
            stream.forEach(p -> {
                try {
                    this.deleteAll((Path)p, parts, index + 1, exclusions);
                }
                catch (IOException e) {
                    LOGGER.warn("Exception during delete of " + String.valueOf(this.rootDir.relativize((Path)p)), (Throwable)e);
                }
            });
        }
    }

    private void deleteIndex(String name) throws IOException, InterruptedException {
        this.unload(name, true);
    }

    public void setMaxIndexesOpen(int maxIndexesOpen) {
        this.maxIndexesOpen = maxIndexesOpen;
    }

    public void setCommitIntervalSeconds(int commitIntervalSeconds) {
        this.commitIntervalSeconds = commitIntervalSeconds;
    }

    public void setIdleSeconds(int idleSeconds) {
        this.idleSeconds = idleSeconds;
    }

    public void setRootDir(Path rootDir) {
        this.rootDir = rootDir;
    }

    public void setObjectMapper(ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
    }

    public void setScheduledExecutorService(ScheduledExecutorService schedulerExecutorService) {
        this.schedulerExecutorService = schedulerExecutorService;
    }

    public void setSearcherFactory(SearcherFactory searcherFactory) {
        this.searcherFactory = searcherFactory;
    }

    public void start() throws IOException {
        Objects.requireNonNull(this.objectMapper, "objectMapper must not be null");
        Objects.requireNonNull(this.rootDir, "rootDir must not be null");
        Objects.requireNonNull(this.searcherFactory, "searcherFactory must not be null");
        Objects.requireNonNull(this.schedulerExecutorService, "schedulerExecutorService must not be null");
        this.cache = new LRUMap(this.maxIndexesOpen);
        this.createLock = new StripedLock(100);
    }

    private Runnable commitFun(String name, IndexHolder holder) {
        return () -> {
            holder.lock.readLock().lock();
            try {
                if (holder.index.commit()) {
                    LOGGER.info("committed {}", (Object)name);
                }
            }
            catch (IOException e) {
                LOGGER.warn("I/O exception while committing " + name, (Throwable)e);
            }
            finally {
                holder.lock.readLock().unlock();
            }
        };
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void stop() throws IOException, InterruptedException {
        LRUMap lRUMap = this.cache;
        synchronized (lRUMap) {
            HashSet names = new HashSet(this.cache.keySet());
            for (String name : names) {
                this.unload(name, false);
            }
            this.cache.clear();
        }
    }

    private boolean isIndex(Path path) {
        return path.resolve("index_definition.json").toFile().exists();
    }

    private Path indexDefinitionPath(String name) {
        return this.indexRootPath(name).resolve("index_definition.json");
    }

    private Path indexPath(String name) {
        return this.indexRootPath(name).resolve("index");
    }

    private IndexDefinition loadIndexDefinition(String name) throws IOException {
        return (IndexDefinition)this.objectMapper.readValue(this.indexDefinitionPath(name).toFile(), IndexDefinition.class);
    }

    private Path indexRootPath(String name) {
        Path result = this.rootDir.resolve(name);
        if (result.startsWith(this.rootDir)) {
            return result;
        }
        throw new WebApplicationException(name + " attempts to escape from index root directory", Response.Status.BAD_REQUEST);
    }

    private Index load(String name) throws IOException {
        LOGGER.info("opening {}", (Object)name);
        Path path = this.indexPath(name);
        IndexDefinition indexDefinition = this.loadIndexDefinition(name);
        Analyzer analyzer = Lucene9AnalyzerFactory.fromDefinition(indexDefinition);
        DirectIODirectory dir = new DirectIODirectory(FSDirectory.open((Path)path.resolve("9")));
        IndexWriterConfig config = new IndexWriterConfig(analyzer);
        config.setUseCompoundFile(false);
        IndexWriter writer = new IndexWriter((Directory)dir, config);
        long updateSeq = this.getSeq(writer, "update_seq");
        long purgeSeq = this.getSeq(writer, "purge_seq");
        SearcherManager searcherManager = new SearcherManager(writer, this.searcherFactory);
        return new Lucene9Index(analyzer, writer, updateSeq, purgeSeq, searcherManager);
    }

    private long getSeq(IndexWriter writer, String key) throws IOException {
        Iterable commitData = writer.getLiveCommitData();
        if (commitData == null) {
            return 0L;
        }
        for (Map.Entry entry : commitData) {
            if (!((String)entry.getKey()).equals(key)) continue;
            return Long.parseLong((String)entry.getValue());
        }
        return 0L;
    }

    private void close(String name, IndexHolder holder) throws IOException {
        if (!holder.lock.isWriteLockedByCurrentThread()) {
            throw new IllegalStateException();
        }
        holder.commitFuture.cancel(true);
        IOUtils.runAll(() -> {
            if (holder.index.commit()) {
                LOGGER.debug("committed {} before close", (Object)name);
            }
        }, () -> holder.index.close(), () -> {
            if (holder.index.isDeleteOnClose()) {
                IOUtils.rm(this.indexRootPath(name));
            }
        });
    }

    private void assertSame(IndexDefinition a, IndexDefinition b) {
        if (!a.equals(b)) {
            throw new WebApplicationException("Index already exists", Response.Status.EXPECTATION_FAILED);
        }
    }

    private class LRUMap
    extends LinkedHashMap<String, IndexHolder> {
        private LRUMap(int capacity) {
            super(capacity, 0.75f, true);
        }
    }

    private static class IndexHolder {
        private final ReentrantReadWriteLock lock = new ReentrantReadWriteLock(true);
        private ScheduledFuture<?> commitFuture;
        private HolderState state = HolderState.NOT_LOADED;
        private Index index;

        private IndexHolder() {
        }
    }

    private static enum HolderState {
        NOT_LOADED,
        LOADED,
        UNLOADED;

    }

    @FunctionalInterface
    public static interface IndexFunction<V, R> {
        public R apply(V var1) throws IOException;
    }
}

